#ifndef DM_LINUX_INTERNAL_H
#define DM_LINUX_INTERNAL_H

#include "dm_nutypedefs.h"
#include <linux/kernel.h>
#include <linux/kfifo.h>
#include <linux/delay.h>
#include <linux/semaphore.h>
#include <linux/spinlock.h>
#include <linux/sched.h>

/**
 * This is only meant to contain functions and structures that are needed
 * across multiple translation units for the linux dm implementation.
 */

/*
 * Code in here will break because it assumes
 *sizeof(pointer) == sizeof(UINT32).
 * Standardizing on UINT32 may be sub-optimal, since it could cause problems
 *when moving to a 64-bit OS.
 */
typedef UINT32 DM_RESOURCE_HANDLE;

typedef enum {
    DM_LINUX_EVENT_TYPE_SINGLE,
    DM_LINUX_EVENT_TYPE_GROUP
} DmLinuxEventType;

struct LinuxEvent
{
    DmLinuxEventType type;
    wait_queue_head_t wq;
    //TODO: This could almost certainly be replaced by spinlocks
    //struct mutex m;
    spinlock_t slock;
    UINT32 events;
};

static int eventErrors(struct LinuxEvent *le, DmLinuxEventType type)
{
    int err = 0;
    if(!le) {
        pr_err("Usage ERROR:------------ NULL Event.\n");
        err = -EFAULT;
    }
    else if(le->type != type) {
        pr_err("Usage ERROR:------------ Wrong Event Type.\n");
        err = -EINVAL;
    }
    BUG_ON(err);
    return err;
}


/**
 * Need a bookkeeping structure, since the Dm fifos use fixed-size objects.
 * Note that these fifos give a minimum amount of guarantees. Locking is
 * done via spinlocks that don't spin on uniprocessor systems.
 */
struct LinuxQueue
{
    size_t objectSize; /* Store the object size, since OSQueue doesn't accept a size argument. */
    /* TODO: Could do something smarter here, and allocate the type of kfifo
     * depending on the object size. Instead, play it safe and allow objects up
     * to 65535 bytes in size.
     */
    spinlock_t slock;
    struct kfifo_rec_ptr_2 fifo;
    wait_queue_head_t wq;
};

#define PRINT_QUEUE(q) \
({ \
    size_t size = q ? kfifo_size(&q->fifo) : 0; \
    size_t used = q ? kfifo_len(&q->fifo) : 0; \
    pr_info("%s: queue = 0x%p objectSize = %6u fullness=%u/%u = %u\n", \
            __FUNCTION__, q, q ? q->objectSize : 0, \
            used, size, used/size); \
})
//#define PRINT_QUEUE(q)

static inline int checkQueue(struct LinuxQueue *q)
{
    WARN_ON_ONCE(!q);
    return !q;
}

#endif //DM_LINUX_INTERNAL_H
